/* ----------------------------------------------------------------------------
 *         ATMEL Microcontroller Software Support  -  ROUSSET  -
 * ----------------------------------------------------------------------------
 * Copyright (c) 2006, Atmel Corporation

 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer below.
 *
 * - Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the disclaimer below in the documentation and/or
 * other materials provided with the distribution.
 *
 * Atmel's name may not be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * DISCLAIMER: THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ----------------------------------------------------------------------------
 */

//------------------------------------------------------------------------------
//         Headers
//------------------------------------------------------------------------------

#include "sdmmc.h"
#include <board.h>
#include <utility/assert.h>
#include <utility/trace.h>

#include <string.h>

//------------------------------------------------------------------------------
//         Local constants
//------------------------------------------------------------------------------

// SD card operation states
#define SD_STATE_STBY     0
#define SD_STATE_DATA     1
#define SD_STATE_RCV      2

// Card type
#define UNKNOWN_CARD   0
#define CARD_SD        1
#define CARD_SDHC      2
#define CARD_MMC       3

// Delay between sending MMC commands
#define MMC_DELAY 0x4ff

#define SD_ADDRESS(pSd, address) (((pSd)->cardType == CARD_SDHC) ? \
                                 (address):((address) << SD_BLOCK_SIZE_BIT))

// Status register constants
#define STATUS_READY_FOR_DATA   (1 << 8)
#define STATUS_IDLE             (0 << 9)
#define STATUS_READY            (1 << 9)
#define STATUS_IDENT            (2 << 9)
#define STATUS_STBY             (3 << 9)
#define STATUS_TRAN             (4 << 9)
#define STATUS_DATA             (5 << 9)
#define STATUS_RCV              (6 << 9)
#define STATUS_PRG              (7 << 9)
#define STATUS_DIS              (8 << 9)
#define STATUS_STATE            (0xF << 9)

////////////////////////////////////////////////////////////////////////////////////////////////////
// OCR Register
////////////////////////////////////////////////////////////////////////////////////////////////////
#define AT91C_VDD_16_17                 (1UL << 4)
#define AT91C_VDD_17_18                 (1UL << 5)
#define AT91C_VDD_18_19                 (1UL << 6)
#define AT91C_VDD_19_20                 (1UL << 7)
#define AT91C_VDD_20_21                 (1UL << 8)
#define AT91C_VDD_21_22                 (1UL << 9)
#define AT91C_VDD_22_23                 (1UL << 10)
#define AT91C_VDD_23_24                 (1UL << 11)
#define AT91C_VDD_24_25                 (1UL << 12)
#define AT91C_VDD_25_26                 (1UL << 13)
#define AT91C_VDD_26_27                 (1UL << 14)
#define AT91C_VDD_27_28                 (1UL << 15)
#define AT91C_VDD_28_29                 (1UL << 16)
#define AT91C_VDD_29_30                 (1UL << 17)
#define AT91C_VDD_30_31                 (1UL << 18)
#define AT91C_VDD_31_32                 (1UL << 19)
#define AT91C_VDD_32_33                 (1UL << 20)
#define AT91C_VDD_33_34                 (1UL << 21)
#define AT91C_VDD_34_35                 (1UL << 22)
#define AT91C_VDD_35_36                 (1UL << 23)
#define AT91C_CARD_POWER_UP_BUSY        (1UL << 31)

#define AT91C_MMC_HOST_VOLTAGE_RANGE     (AT91C_VDD_27_28 +\
                                          AT91C_VDD_28_29 +\
                                          AT91C_VDD_29_30 +\
                                          AT91C_VDD_30_31 +\
                                          AT91C_VDD_31_32 +\
                                          AT91C_VDD_32_33)


#define AT91C_CCS    (1UL << 30)

#define AT91C_NO_ARGUMENT       0x0

/////////////////////////////////////////////////////////////////////////////////////////////////////
// MCI_CMD Register Value
/////////////////////////////////////////////////////////////////////////////////////////////////////
#define AT91C_POWER_ON_INIT         (0  | AT91C_MCI_TRCMD_NO    | AT91C_MCI_SPCMD_INIT | AT91C_MCI_OPDCMD)

/////////////////////////////////////////////////////////////////
// Class 0 & 1 commands: Basic commands and Read Stream commands
/////////////////////////////////////////////////////////////////

#define AT91C_GO_IDLE_STATE_CMD         (0  | AT91C_MCI_TRCMD_NO    | AT91C_MCI_SPCMD_NONE )
#define AT91C_MMC_GO_IDLE_STATE_CMD     (0  | AT91C_MCI_TRCMD_NO    | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_OPDCMD)
#define AT91C_MMC_SEND_OP_COND_CMD      (1  | AT91C_MCI_TRCMD_NO    | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48 | AT91C_MCI_OPDCMD)
#define AT91C_ALL_SEND_CID_CMD          (2  | AT91C_MCI_TRCMD_NO    | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_136 )
#define AT91C_MMC_ALL_SEND_CID_CMD      (2  | AT91C_MCI_TRCMD_NO    | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_136 | AT91C_MCI_OPDCMD)
#define AT91C_SET_RELATIVE_ADDR_CMD     (3  | AT91C_MCI_TRCMD_NO    | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48  | AT91C_MCI_MAXLAT )
#define AT91C_MMC_SET_RELATIVE_ADDR_CMD (3  | AT91C_MCI_TRCMD_NO    | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48  | AT91C_MCI_MAXLAT | AT91C_MCI_OPDCMD)

#define AT91C_SET_DSR_CMD           (4  | AT91C_MCI_TRCMD_NO    | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_NO       | AT91C_MCI_MAXLAT )    // not tested

#define AT91C_SEL_DESEL_CARD_CMD            (7  | AT91C_MCI_TRCMD_NO    | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_MAXLAT )
#define AT91C_SEND_IF_COND                  (8  | AT91C_MCI_TRCMD_NO    | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48 | AT91C_MCI_MAXLAT)
#define AT91C_SEND_CSD_CMD                  (9  | AT91C_MCI_TRCMD_NO    | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_136  | AT91C_MCI_MAXLAT )
#define AT91C_SEND_CID_CMD                  (10 | AT91C_MCI_TRCMD_NO    | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_136  | AT91C_MCI_MAXLAT )
#define AT91C_MMC_READ_DAT_UNTIL_STOP_CMD   (11 | AT91C_MCI_TRTYP_STREAM| AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRDIR   | AT91C_MCI_TRCMD_START | AT91C_MCI_MAXLAT )

#define AT91C_STOP_TRANSMISSION_CMD         (12 | AT91C_MCI_TRCMD_STOP  | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_MAXLAT )
#define AT91C_STOP_TRANSMISSION_SYNC_CMD    (12 | AT91C_MCI_TRCMD_STOP  | AT91C_MCI_SPCMD_SYNC  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_MAXLAT )
#define AT91C_SEND_STATUS_CMD               (13 | AT91C_MCI_TRCMD_NO    | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_MAXLAT )
#define AT91C_GO_INACTIVE_STATE_CMD         (15 | AT91C_MCI_RSPTYP_NO )

//*------------------------------------------------
//* Class 2 commands: Block oriented Read commands
//*------------------------------------------------

#define AT91C_SET_BLOCKLEN_CMD          (16 | AT91C_MCI_TRCMD_NO    | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_MAXLAT )
#define AT91C_READ_SINGLE_BLOCK_CMD     (17 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_START | AT91C_MCI_TRTYP_BLOCK | AT91C_MCI_TRDIR   | AT91C_MCI_MAXLAT)
#define AT91C_READ_MULTIPLE_BLOCK_CMD   (18 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_START | AT91C_MCI_TRTYP_MULTIPLE  | AT91C_MCI_TRDIR   | AT91C_MCI_MAXLAT)

//*--------------------------------------------
//* Class 3 commands: Sequential write commands
//*--------------------------------------------

#define AT91C_MMC_WRITE_DAT_UNTIL_STOP_CMD  (20 | AT91C_MCI_TRTYP_STREAM| AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48 & ~(AT91C_MCI_TRDIR) | AT91C_MCI_TRCMD_START | AT91C_MCI_MAXLAT ) // MMC

//*------------------------------------------------
//* Class 4 commands: Block oriented write commands
//*------------------------------------------------

#define AT91C_WRITE_BLOCK_CMD           (24 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_START | (AT91C_MCI_TRTYP_BLOCK    &  ~(AT91C_MCI_TRDIR))  | AT91C_MCI_MAXLAT)
#define AT91C_WRITE_MULTIPLE_BLOCK_CMD  (25 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_START | (AT91C_MCI_TRTYP_MULTIPLE &  ~(AT91C_MCI_TRDIR))  | AT91C_MCI_MAXLAT)
#define AT91C_PROGRAM_CSD_CMD           (27 | AT91C_MCI_RSPTYP_48 )


//*----------------------------------------
//* Class 6 commands: Group Write protect
//*----------------------------------------

#define AT91C_SET_WRITE_PROT_CMD        (28 | AT91C_MCI_RSPTYP_48 )
#define AT91C_CLR_WRITE_PROT_CMD        (29 | AT91C_MCI_RSPTYP_48 )
#define AT91C_SEND_WRITE_PROT_CMD       (30 | AT91C_MCI_RSPTYP_48 )


//*----------------------------------------
//* Class 5 commands: Erase commands
//*----------------------------------------

#define AT91C_TAG_SECTOR_START_CMD          (32 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_NO    | AT91C_MCI_MAXLAT)
#define AT91C_TAG_SECTOR_END_CMD            (33 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_NO    | AT91C_MCI_MAXLAT)
#define AT91C_MMC_UNTAG_SECTOR_CMD          (34 | AT91C_MCI_RSPTYP_48 )
#define AT91C_MMC_TAG_ERASE_GROUP_START_CMD (35 | AT91C_MCI_RSPTYP_48 )
#define AT91C_MMC_TAG_ERASE_GROUP_END_CMD   (36 | AT91C_MCI_RSPTYP_48 )
#define AT91C_MMC_UNTAG_ERASE_GROUP_CMD     (37 | AT91C_MCI_RSPTYP_48 )
#define AT91C_ERASE_CMD                     (38 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_NO    | AT91C_MCI_MAXLAT )

//*----------------------------------------
//* Class 7 commands: Lock commands
//*----------------------------------------

#define AT91C_LOCK_UNLOCK           (42 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_NO    | AT91C_MCI_MAXLAT) // not tested

//*-----------------------------------------------
// Class 8 commands: Application specific commands
//*-----------------------------------------------

#define AT91C_APP_CMD               (55 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_NO | AT91C_MCI_MAXLAT)
#define AT91C_GEN_CMD               (56 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_NO | AT91C_MCI_MAXLAT)    // not tested

#define AT91C_SDCARD_SET_BUS_WIDTH_CMD          (6  | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_NO    | AT91C_MCI_MAXLAT)
#define AT91C_SDCARD_STATUS_CMD                 (13 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_START | AT91C_MCI_TRTYP_BLOCK | AT91C_MCI_TRDIR | AT91C_MCI_MAXLAT)
#define AT91C_SDCARD_SEND_NUM_WR_BLOCKS_CMD     (22 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_NO    | AT91C_MCI_MAXLAT)
#define AT91C_SDCARD_SET_WR_BLK_ERASE_COUNT_CMD (23 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_NO    | AT91C_MCI_MAXLAT)
#define AT91C_SDCARD_APP_OP_COND_CMD            (41 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_NO )
#define AT91C_SDCARD_SET_CLR_CARD_DETECT_CMD    (42 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_NO    | AT91C_MCI_MAXLAT)
#define AT91C_SDCARD_SEND_SCR_CMD               (51 | AT91C_MCI_SPCMD_NONE  | AT91C_MCI_RSPTYP_48   | AT91C_MCI_TRCMD_NO    | AT91C_MCI_MAXLAT)

#define AT91C_SDCARD_APP_ALL_CMD                (AT91C_SDCARD_SET_BUS_WIDTH_CMD +\
                                                                AT91C_SDCARD_STATUS_CMD +\
                                                                AT91C_SDCARD_SEND_NUM_WR_BLOCKS_CMD +\
                                                                AT91C_SDCARD_SET_WR_BLK_ERASE_COUNT_CMD +\
                                                                AT91C_SDCARD_APP_OP_COND_CMD +\
                                                                AT91C_SDCARD_SET_CLR_CARD_DETECT_CMD +\
                                                                AT91C_SDCARD_SEND_SCR_CMD)

//*----------------------------------------
//* Class 9 commands: IO Mode commands
//*----------------------------------------

#define AT91C_MMC_FAST_IO_CMD                   (39 | AT91C_MCI_SPCMD_NONE | AT91C_MCI_RSPTYP_48 | AT91C_MCI_MAXLAT)
#define AT91C_MMC_GO_IRQ_STATE_CMD              (40 | AT91C_MCI_SPCMD_NONE | AT91C_MCI_RSPTYP_48 | AT91C_MCI_TRCMD_NO   | AT91C_MCI_MAXLAT)


//------------------------------------------------------------------------------
//         Local functions
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
/// Sends the current SD card driver command to the card.
/// Returns 0 if successful; Otherwise, returns the transfer status code or
/// SD_ERROR_MCI if there was a problem with the MCI transfer.
/// \param pSd  Pointer to a SdCard driver instance.
//------------------------------------------------------------------------------
static unsigned char SendCommand(SdCard *pSd)
{
    MciCmd *pCommand = &(pSd->command);
    unsigned char error;

    // Send command
    error = MCI_SendCommand(pSd->pMci, pCommand);
    if (error) {

        trace_LOG(trace_ERROR, "-E- SendCommand: Failed to send command (%d)\n\r", error);
        return SD_ERROR_MCI;
    }

    // Wait for command to complete
    while (!MCI_IsTxComplete(pCommand));

    if(pCommand->cmd == AT91C_STOP_TRANSMISSION_CMD) {
        while (MCI_CheckBusy(pSd->pMci) != 0);
    }

    // Delay between sending commands, only for MMC card test.
    if((pSd->cardType == CARD_MMC)||(pSd->cardType == UNKNOWN_CARD)) {
        int i;
        for(i=0; i < MMC_DELAY; i++);
    }

    return pCommand->status;
}

//------------------------------------------------------------------------------
/// Initialization delay: The maximum of 1 msec, 74 clock cycles and supply ramp
/// up time.
/// Returns the command transfer result (see SendCommand).
/// \param pSd  Pointer to a SdCard driver instance.
//------------------------------------------------------------------------------
static unsigned char Pon(SdCard *pSd)
{
    MciCmd *pCommand = &(pSd->command);

    memset(pCommand, 0, sizeof(MciCmd));
   // Fill command information
    pCommand->cmd = AT91C_POWER_ON_INIT;
    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // Send command
    return SendCommand(pSd);
}

//------------------------------------------------------------------------------
/// Resets all cards to idle state.
/// Returns the command transfer result (see SendCommand).
/// \param pSd  Pointer to a SdCard driver instance.
//------------------------------------------------------------------------------
static unsigned char Cmd0(SdCard *pSd)
{
    MciCmd *pCommand = &(pSd->command);

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_GO_IDLE_STATE_CMD;
    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // send command
    return SendCommand(pSd);
}

//------------------------------------------------------------------------------
/// MMC send operation condition command.
/// Returns the command transfer result (see SendCommand).
/// \param pSd  Pointer to a SdCard driver instance.
//------------------------------------------------------------------------------
static unsigned char Cmd1(SdCard *pSd)
{
    MciCmd *pCommand = &(pSd->command);
    unsigned char error;
    unsigned int response;

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_MMC_SEND_OP_COND_CMD;
    pCommand->arg = AT91C_MMC_HOST_VOLTAGE_RANGE;
    pCommand->resSize = 1;
    pCommand->pResp = &response;

    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // send command
    error = SendCommand(pSd);
    if (error) {

        return error;
    }

    if ((response & AT91C_CARD_POWER_UP_BUSY) == AT91C_CARD_POWER_UP_BUSY) {
        return 0;
    }
    else {
        return SD_ERROR_MCI;
    }
}

//------------------------------------------------------------------------------
/// Asks to all cards to send the CID numbers.
/// Returns the command transfer result (see SendCommand).
/// \param pSd  Pointer to a SD card driver instance.
/// \param pCid  Buffer for storing the CID numbers.
//------------------------------------------------------------------------------
static unsigned char Cmd2(SdCard *pSd, unsigned int *pCid)
{
    MciCmd *pCommand = &(pSd->command);

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill the command information
    pCommand->cmd = AT91C_ALL_SEND_CID_CMD;
    pCommand->resSize = 4;
    pCommand->pResp = pCid;
    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // Send the command
    return SendCommand(pSd);
}

//------------------------------------------------------------------------------
/// Asks The card to publish a new relative address.
/// Returns the command transfer result (see SendCommand).
/// \param pSd  Pointer to a SD card driver instance.
//------------------------------------------------------------------------------
static unsigned char Cmd3(SdCard *pSd)
{
    MciCmd *pCommand = &(pSd->command);
    unsigned int cardAddress;
    unsigned char error;

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_SET_RELATIVE_ADDR_CMD;
    // Assign relative address to MMC card
    if (pSd->cardType == CARD_MMC) {
        pCommand->arg = (0x1 << 16);
    }
    pCommand->resSize = 1;
    pCommand->pResp = &cardAddress;
    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // Send command
    error = SendCommand(pSd);
    if (error) {

        return error;
    }

    // Save card address in driver
    if (pSd->cardType != CARD_MMC) {
        pSd->cardAddress = (cardAddress >> 16) & 0xFFFF;
    }
    else {
        // Default MMC RCA is 0x0001
        pSd->cardAddress = 1;
    }

    return 0;
}

//------------------------------------------------------------------------------
/// Toggles a card between the stand-by and the transfer states or between the
/// programming and disconnects states.
/// Returns the command transfer result (see SendCommand).
/// \param pSd  Pointer to a SD card driver instance.
/// \param address  Relative Card Address (0 deselects all).
//------------------------------------------------------------------------------
static unsigned char Cmd7(SdCard *pSd, unsigned int address)
{
    MciCmd *pCommand = &(pSd->command);

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_SEL_DESEL_CARD_CMD;
    pCommand->arg = address << 16;
    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // Send command
    return SendCommand(pSd);
}

//------------------------------------------------------------------------------
/// Voltage check.
/// Returns 0 if successful; otherwise returns SD_ERROR_NORESPONSE if the card did
/// not answer the command, or SD_ERROR_MCI.
/// \param pSd  Pointer to a SD card driver instance.
/// \param supplyVoltage  Expected supply voltage.
//------------------------------------------------------------------------------
static unsigned char Cmd8(SdCard *pSd, unsigned char supplyVoltage)
{
    MciCmd *pCommand = &(pSd->command);
    unsigned int response;
    unsigned char error;

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_SEND_IF_COND;
    pCommand->arg = (supplyVoltage << 8) | (0xAA);
    pCommand->resSize    = 1;
    pCommand->pResp      = &response;
    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // Send command
    error = SendCommand(pSd);

    // Check result
    if (error == MCI_STATUS_NORESPONSE) {

        return SD_ERROR_NORESPONSE;
    }
    else if (!error && (response == ((supplyVoltage << 8) | 0xAA))) {

        return 0;
    }
    else {

        return SD_ERROR_MCI;
    }
}

//------------------------------------------------------------------------------
/// Addressed card sends its card specific data.
/// Returns the command transfer result (see SendCommand).
/// \param pSd  Pointer to a SD card driver instance.
//------------------------------------------------------------------------------
static unsigned char Cmd9(SdCard *pSd)
{
    MciCmd *pCommand = &(pSd->command);

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_SEND_CSD_CMD;
    pCommand->arg = pSd->cardAddress << 16;
    pCommand->resSize = 4;
    pCommand->pResp = pSd->csd;
    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // Send command
    return SendCommand(pSd);
}

//------------------------------------------------------------------------------
/// Forces the card to stop transmission.
/// \param pSd  Pointer to a SD card driver instance.
/// \param pStatus  Pointer to a status variable.
//------------------------------------------------------------------------------
static unsigned char Cmd12(SdCard *pSd)
{
    MciCmd *pCommand = &(pSd->command);

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_STOP_TRANSMISSION_CMD;
    pCommand->conTrans = MCI_NEW_TRANSFER;
    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // Send command
    return SendCommand(pSd);
}

//------------------------------------------------------------------------------
/// Addressed card sends its status register.
/// Returns the command transfer result (see SendCommand).
/// \param pSd  Pointer to a SD card driver instance.
/// \param pStatus  Pointer to a status variable.
//------------------------------------------------------------------------------
static unsigned char Cmd13(SdCard *pSd, unsigned int *pStatus)
{
    MciCmd *pCommand = &(pSd->command);

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_SEND_STATUS_CMD;
    pCommand->arg = pSd->cardAddress << 16;
    pCommand->resSize = 1;
    pCommand->pResp = pStatus;
    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // Send command
    return SendCommand(pSd);
}

//------------------------------------------------------------------------------
/// In the case of a Standard Capacity SD Memory Card, this command sets the
/// block length (in bytes) for all following block commands (read, write, lock).
/// Default block length is fixed to 512 Bytes.
/// Set length is valid for memory access commands only if partial block read
/// operation are allowed in CSD.
/// In the case of a High Capacity SD Memory Card, block length set by CMD16
/// command does not affect the memory read and write commands. Always 512
/// Bytes fixed block length is used. This command is effective for LOCK_UNLOCK command.
/// In both cases, if block length is set larger than 512Bytes, the card sets the
/// BLOCK_LEN_ERROR bit.
/// \param pSd  Pointer to a SD card driver instance.
/// \param blockLength  Block length in bytes.
//------------------------------------------------------------------------------
static unsigned char Cmd16(SdCard *pSd, unsigned short blockLength)
{
    MciCmd *pCommand = &(pSd->command);

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_SET_BLOCKLEN_CMD;
    pCommand->arg = blockLength;
    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // Send command
    return SendCommand(pSd);
}

//------------------------------------------------------------------------------
/// In the case of a Standard Capacity SD Memory Card, this command reads a block
/// of the size selected by the SET_BLOCKLEN.
/// In the case of a high capacity card, block length is fixed 512 bytes.
/// \param pSd  Pointer to a SD card driver instance.
/// \param blockSize  Block size (shall be set to 512 in case of high capacity).
/// \param pData  Pointer to the application buffer to be filled.
/// \param address  SD card address.
//------------------------------------------------------------------------------
static unsigned char Cmd17(
    SdCard *pSd,
    unsigned char *pData,
    unsigned int address)
{
    MciCmd *pCommand = &(pSd->command);

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_READ_SINGLE_BLOCK_CMD;
    pCommand->arg = address;
    pCommand->blockSize = SD_BLOCK_SIZE;
    pCommand->pData = pData;
    pCommand->isRead = 1;
    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // Send command
    return SendCommand(pSd);
}

//------------------------------------------------------------------------------
/// Continously transfers datablocks from card to host until interrupted by a
/// STOP_TRANSMISSION command.
/// \param pSd  Pointer to a SD card driver instance.
/// \param blockSize  Block size (shall be set to 512 in case of high capacity).
/// \param pData  Pointer to the application buffer to be filled.
/// \param address  SD card address.
//------------------------------------------------------------------------------
static unsigned char Cmd18(
    SdCard *pSd,
    unsigned short nbBlock,
    unsigned char *pData,
    unsigned int address)
{
    MciCmd *pCommand = &(pSd->command);

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_READ_MULTIPLE_BLOCK_CMD;
    pCommand->arg = address;
    pCommand->blockSize = SD_BLOCK_SIZE;
    pCommand->nbBlock = nbBlock;
    pCommand->pData = pData;
    pCommand->isRead = 1;
    pCommand->conTrans = MCI_NEW_TRANSFER;
    // Set SD command state
    pSd->state = SD_STATE_DATA;

    // Send command
    return SendCommand(pSd);
}

//------------------------------------------------------------------------------
/// Write block command
/// \param pSd  Pointer to a SD card driver instance.
/// \param blockSize  Block size (shall be set to 512 in case of high capacity).
/// \param pData  Pointer to the application buffer to be filled.
/// \param address  SD card address.
//------------------------------------------------------------------------------
static unsigned char Cmd24(
    SdCard *pSd,
    unsigned short blockSize,
    const unsigned char *pData,
    unsigned int address)
{
    MciCmd *pCommand = &(pSd->command);

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_WRITE_BLOCK_CMD;
    pCommand->arg = address;
    pCommand->blockSize = blockSize;
    pCommand->pData = (unsigned char *) pData;
    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // Send command
    return SendCommand(pSd);
}

//------------------------------------------------------------------------------
/// Write block command
/// \param pSd  Pointer to a SD card driver instance.
/// \param blockSize  Block size (shall be set to 512 in case of high capacity).
/// \param pData  Pointer to the application buffer to be filled.
/// \param address  SD card address.
//------------------------------------------------------------------------------
static unsigned char Cmd25(
    SdCard *pSd,
    unsigned short nbBlock,
    unsigned char *pData,
    unsigned int address)
{
    MciCmd *pCommand = &(pSd->command);

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_WRITE_MULTIPLE_BLOCK_CMD;
    pCommand->arg = address;
    pCommand->blockSize = SD_BLOCK_SIZE;
    pCommand->nbBlock = nbBlock;
    pCommand->pData = (unsigned char *) pData;
    pCommand->conTrans = MCI_NEW_TRANSFER;
    // Set SD command state
    pSd->state = SD_STATE_RCV;

    // Send command
    return SendCommand(pSd);
}


//------------------------------------------------------------------------------
/// Initialization delay: The maximum of 1 msec, 74 clock cycles and supply
/// ramp up time.
/// Returns the command transfer result (see SendCommand).
/// \param pSd  Pointer to a SD card driver instance.
//------------------------------------------------------------------------------
static unsigned char Cmd55(SdCard *pSd)
{
    MciCmd *pCommand = &(pSd->command);

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_APP_CMD;
    pCommand->arg = (pSd->cardAddress << 16);
    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // Send command
    return SendCommand(pSd);
}

//------------------------------------------------------------------------------
/// Defines the data bus width (00=1bit or 10=4 bits bus) to be used for data transfer.
/// The allowed data bus widths are given in SCR register.
/// Returns the command transfer result (see SendCommand).
/// \param pSd  Pointer to a SD card driver instance.
/// \param busWidth  Bus width in bits.
//------------------------------------------------------------------------------
static unsigned char Acmd6(SdCard *pSd, unsigned char busWidth)
{
    MciCmd *pCommand = &(pSd->command);
    unsigned char error;

    // Delay
    error = Cmd55(pSd);
    if (error) {

        return error;
    }

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_SDCARD_SET_BUS_WIDTH_CMD;
    if (busWidth == 4) {

        pCommand->arg = SD_SCR_BUS_WIDTH_4BITS;
    }
    else {

        pCommand->arg = SD_SCR_BUS_WIDTH_1BIT;
    }
    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // Send command
    return SendCommand(pSd);
}

//------------------------------------------------------------------------------
/// Asks to all cards to send their operations conditions.
/// Returns the command transfer result (see SendCommand).
/// \param pSd  Pointer to a SD card driver instance.
/// \param hcs  Shall be true if Host support High capacity.
/// \param pCCS  Set the pointed flag to 1 if hcs != 0 and SD OCR CCS flag is set.
//------------------------------------------------------------------------------
static unsigned char Acmd41(SdCard *pSd, unsigned char hcs, unsigned char *pCCS)
{
    MciCmd *pCommand = &(pSd->command);
    unsigned char error;
    unsigned int response;

    do {
        // Delay
        error = Cmd55(pSd);
        if (error) {

            return error;
        }

        memset(pCommand, 0, sizeof(MciCmd));
        // Fill command information
        pCommand->cmd = AT91C_SDCARD_APP_OP_COND_CMD;
        pCommand->arg = AT91C_MMC_HOST_VOLTAGE_RANGE;
        if (hcs) {

            pCommand->arg |= AT91C_CCS;
        }
        pCommand->resSize = 1;
        pCommand->pResp = &response;

        // Set SD command state
        pSd->state = SD_STATE_STBY;

        // Send command
        error = SendCommand(pSd);
        if (error) {

            return error;
        }

        *pCCS  = ((response & AT91C_CCS) != 0);
    }
    while ((response & AT91C_CARD_POWER_UP_BUSY) != AT91C_CARD_POWER_UP_BUSY);

    return 0;
}

//------------------------------------------------------------------------------
/// Read the SD Configuration register.
/// \param pSd  Pointer to a SD card driver instance.
/// \param pSdScr  Variable for storing the configuration register.
//------------------------------------------------------------------------------
static unsigned char Acmd51(SdCard *pSd, unsigned int *pSdScr)
{
    MciCmd *pCommand = &(pSd->command);
    unsigned char error;

    // Delay
    error = Cmd55(pSd);
    if (error) {

        return error;
    }

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->cmd = AT91C_SDCARD_SEND_SCR_CMD;
    pCommand->resSize = 2;
    pCommand->pResp = pSdScr;
    // Set SD command state
    pSd->state = SD_STATE_STBY;

    // Send command
    return SendCommand(pSd);
}

//------------------------------------------------------------------------------
/// Continue to transfer datablocks from card to host until interrupted by a
/// STOP_TRANSMISSION command.
/// \param pSd  Pointer to a SD card driver instance.
/// \param blockSize  Block size (shall be set to 512 in case of high capacity).
/// \param pData  Pointer to the application buffer to be filled.
/// \param address  SD card address.
//------------------------------------------------------------------------------
static unsigned char ContinuousRead(
    SdCard *pSd,
    unsigned short nbBlock,
    unsigned char *pData,
    unsigned int address)
{
    MciCmd *pCommand = &(pSd->command);

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->blockSize = SD_BLOCK_SIZE;
    pCommand->nbBlock = nbBlock;
    pCommand->pData = pData;
    pCommand->isRead = 1;
    pCommand->conTrans = MCI_CONTINUE_TRANSFER;
    // Set SD command state
    pSd->state = SD_STATE_DATA;

    // Send command
    return SendCommand(pSd);
}

//------------------------------------------------------------------------------
/// Continue to transfer datablocks from host to card until interrupted by a
/// STOP_TRANSMISSION command.
/// \param pSd  Pointer to a SD card driver instance.
/// \param blockSize  Block size (shall be set to 512 in case of high capacity).
/// \param pData  Pointer to the application buffer to be filled.
/// \param address  SD card address.
//------------------------------------------------------------------------------
static unsigned char ContinuousWrite(
    SdCard *pSd,
    unsigned short nbBlock,
    const unsigned char *pData,
    unsigned int address)
{
    MciCmd *pCommand = &(pSd->command);

    memset(pCommand, 0, sizeof(MciCmd));
    // Fill command information
    pCommand->blockSize = SD_BLOCK_SIZE;
    pCommand->nbBlock = nbBlock;
    pCommand->pData = (unsigned char *) pData;
    pCommand->isRead = 0;
    pCommand->conTrans = MCI_CONTINUE_TRANSFER;
    // Set SD command state
    pSd->state = SD_STATE_RCV;

    // Send command
    return SendCommand(pSd);
}
//------------------------------------------------------------------------------
//         Global functions
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
/// Move SD card to transfer state. The buffer size must be at
/// least 512 byte long. This function checks the SD card status register and
/// address the card if required before sending the transfer command.
/// Returns 0 if successful; otherwise returns an code describing the error.
/// \param pSd  Pointer to a SD card driver instance.
/// \param address  Address of the block to transfer.
/// \param nbBlocks Number of blocks to be transfer.
/// \param pData  Data buffer whose size is at least the block size.
/// \param isRead 1 for read data and 0 for write data.
//------------------------------------------------------------------------------
unsigned char MoveToTransferState(
    SdCard *pSd,
    unsigned int address,
    unsigned short nbBlocks,
    unsigned char *pData,
    unsigned char isRead)
{
    unsigned int status;
    unsigned char error;

    if((pSd->state == SD_STATE_DATA)
        || (pSd->state == SD_STATE_RCV)) {

        error = Cmd12(pSd);
        if (error) {
            return error;
        }
    }
    pSd->preBlock = address + (nbBlocks-1);

    if(isRead) {
        // Wait for card to be ready for data transfers
        do {
            error = Cmd13(pSd, &status);
            if (error) {
                return error;
            }
            if (((status & STATUS_STATE) == STATUS_IDLE)  ||
                ((status & STATUS_STATE) == STATUS_READY)   ||
                ((status & STATUS_STATE) == STATUS_IDENT)) {
                printf("state = %x\n\r", (status & STATUS_STATE) >> 9);
                return SD_ERROR_NOT_INITIALIZED;
            }
            // If the SD card is in sending data state or in receivce data state
            if (((status & STATUS_STATE) == STATUS_RCV) ||
                ((status & STATUS_STATE) == STATUS_DATA) ){

                printf("state = %x\n\r", (status & STATUS_STATE) >> 9);
            }
        }
        while (((status & STATUS_READY_FOR_DATA) == 0) ||
              ((status & STATUS_STATE) != STATUS_TRAN));

        ASSERT((status & STATUS_STATE) == STATUS_TRAN,
            "-E- SD Card can't be configured in transfer state %x\n\r", (status & STATUS_STATE)>>9);

        // Read data
        // Move to Sending data state
        error = Cmd18(pSd, nbBlocks, pData, SD_ADDRESS(pSd,address));
        if (error) {
            return error;
        }
    }
    else {
        // Wait for card to be ready for data transfers
        do {
            error = Cmd13(pSd, &status);
            if (error) {
                return error;
            }
        }
        while ((status & STATUS_READY_FOR_DATA) == 0);

        // If the SD card is in standby state go in transfer state
        if ((status & STATUS_STATE) == STATUS_STBY) {
            // Go into transfer state
            error = Cmd7(pSd, pSd->cardAddress);
            if (error) {
                return error;
            }
        }

        // Move to Sending data state
        error = Cmd25(pSd, nbBlocks, pData, SD_ADDRESS(pSd,address));
        if (error) {
            return error;
        }
    }

    return error;
}

//------------------------------------------------------------------------------
/// Read Block of data in a buffer pointed by pData. The buffer size must be at
/// least 512 byte long. This function checks the SD card status register and
/// address the card if required before sending the read command.
/// Returns 0 if successful; otherwise returns an code describing the error.
/// \param pSd  Pointer to a SD card driver instance.
/// \param address  Address of the block to read.
/// \param nbBlocks Number of blocks to be read.
/// \param pData  Data buffer whose size is at least the block size.
//------------------------------------------------------------------------------
unsigned char SD_ReadBlock(
    SdCard *pSd,
    unsigned int address,
    unsigned short nbBlocks,
    unsigned char *pData)
{
    unsigned char error;

    SANITY_CHECK(pSd);
    SANITY_CHECK(pData);
    SANITY_CHECK(nbBlocks);

    if((pSd->state == SD_STATE_DATA)
        && ((pSd->preBlock + 1) == address)) {

        #if defined(at91rm9200)
        error = Cmd12(pSd);
        if (error) {
            return error;
        }
        #else
        error = ContinuousRead(pSd, nbBlocks, pData, SD_ADDRESS(pSd,address));
        pSd->preBlock = address + (nbBlocks-1);
        #endif
    }
    else {
        error = MoveToTransferState(pSd, address, nbBlocks, pData, 1);
    }
    return error;
}

//------------------------------------------------------------------------------
/// Write Block of data pointed by pData. The buffer size must be at
/// least 512 byte long. This function checks the SD card status register and
/// address the card if required before sending the read command.
/// Returns 0 if successful; otherwise returns an SD_ERROR code.
/// \param pSd  Pointer to a SD card driver instance.
/// \param address  Address of block to write.
/// \param nbBlocks Number of blocks to be read
/// \param pData  Pointer to a 512 bytes buffer to be transfered
//------------------------------------------------------------------------------
unsigned char SD_WriteBlock(
    SdCard *pSd,
    unsigned int address,
    unsigned short nbBlocks,
    const unsigned char *pData)
{
    unsigned char error;

    SANITY_CHECK(pSd);
    SANITY_CHECK(pData);
    SANITY_CHECK(nbBlocks);

    if((pSd->state == SD_STATE_RCV)
        && ((pSd->preBlock + 1) == address)) {

        error = ContinuousWrite(pSd, nbBlocks, pData, SD_ADDRESS(pSd,address));

        pSd->preBlock = address + (nbBlocks-1);
    }
    else {

        error = MoveToTransferState(pSd, address, nbBlocks, \
                    (unsigned char *)pData, 0);
    }
    return error;
}

//------------------------------------------------------------------------------
/// Run the SDcard initialization sequence. This function runs the initialisation
/// procedure and the identification process, then it sets the SD card in transfer
/// state to set the block length and the bus width.
/// Returns 0 if successful; otherwise returns an SD_ERROR code.
/// \param pSd  Pointer to a SD card driver instance.
/// \param pMci  Pointer to MCI driver already initialized with MCI_Init().
//------------------------------------------------------------------------------
unsigned char SD_Init(SdCard *pSd, Mci *pMci)
{
    unsigned int sdCid[4];
    unsigned char isCCSet;
    unsigned int status;
    unsigned char error;
    unsigned char cmd8Retries = 2;
    unsigned char cmd1Retries = 100;

    trace_LOG(trace_DEBUG, "-D- SD_Init()\n\r");

    // Initialize SdCard structure
    pSd->pMci = pMci;
    pSd->cardAddress = 0;
    pSd->preBlock = 0xffffffff;
    pSd->state = SD_STATE_STBY;
    pSd->cardType = UNKNOWN_CARD;
    memset(&(pSd->command), 0, sizeof(MciCmd));

    // Initialization delay: The maximum of 1 msec, 74 clock cycles and supply ramp up time
    // Supply ramp up time provides the time that the power is built up to the operating level (the bus
    // master supply voltage) and the time to wait until the SD card can accept the first command

    // Power On Init Special Command
    trace_LOG(trace_DEBUG, "-D- Pon()\n\r");
    error = Pon(pSd);
    if (error) {

        trace_LOG(trace_ERROR, "-E- Error during initialization (%d)\n\r", error);
        return error;
    }

    // The command GO_IDLE_STATE (CMD0) is the software reset command and sets card into Idle State
    // regardless of the current card state.
    trace_LOG(trace_DEBUG, "-D- Cmd0()\n\r");
    error = Cmd0(pSd);
    if (error) {

        trace_LOG(trace_ERROR, "-E- Error during initialization (%d)\n\r", error);
        return error;
    }

    // After power-on or CMD0, all cards CMD lines are in input mode, waiting for start bit of the next command.
    // The cards are initialized with a default relative card address (RCA=0x0000) and with a default
    // driver stage register setting (lowest speed, highest driving current capability).

    // CMD8 is newly added in the Physical Layer Specification Version 2.00 to support multiple voltage
    // ranges and used to check whether the card supports supplied voltage. The version 2.00 host shall
    // issue CMD8 and verify voltage before card initialization.
    // The host that does not support CMD8 shall supply high voltage range...
    trace_LOG(trace_DEBUG, "-D- Cmd8()\n\r");
    do {
        error = Cmd8(pSd, 1);
    }
    while ((error == SD_ERROR_NORESPONSE) && (cmd8Retries-- > 0));

    if (error == SD_ERROR_NORESPONSE) {

        // No response : Ver2.00 or later SD Memory Card(voltage mismatch)
        // or Ver1.X SD Memory Card
        // or not SD Memory Card

        trace_LOG(trace_DEBUG, "-D- No response to Cmd8\n\r");

        // ACMD41 is a synchronization command used to negotiate the operation voltage range and to poll the
        // cards until they are out of their power-up sequence.
        trace_LOG(trace_DEBUG, "-D- Acmd41()\n\r");
        error = Acmd41(pSd, 0, &isCCSet);
        if (error) {

            // Acmd41 failed : MMC card or unknown card
            error = Cmd0(pSd);
            if (error) {

                trace_LOG(trace_ERROR, "-E- Error during initialization (%d)\n\r", error);
                return error;
            }

            do {
                error = Cmd1(pSd);
            }
            while ((error) && (cmd1Retries-- > 0));
            if (error) {
                trace_LOG(trace_ERROR, "-E- Error during initialization (%d)\n\r", error);
                return error;
            }
            else {
                pSd->cardType = CARD_MMC;
            }
        }
        else {
            if(isCCSet == 0) {
                pSd->cardType = CARD_SD;
            }
        }
    }
    else if (!error) {

        // Valid response : Ver2.00 or later SD Memory Card
        trace_LOG(trace_DEBUG, "-D- Acmd41()\n\r");
        error = Acmd41(pSd, 1, &isCCSet);
        if (error) {

            trace_LOG(trace_ERROR, "-E- Error during initialization (%d)\n\r", error);
            return error;
        }
        if (isCCSet)
            pSd->cardType = CARD_SDHC;
        else
            pSd->cardType = CARD_SD;
    }
    else {
        trace_LOG(trace_ERROR, "-E- Error during initialization (%d)\n\r", error);
        return error;
    }


    // The host then issues the command ALL_SEND_CID (CMD2) to the card to get its unique card identification (CID) number.
    // Card that is unidentified (i.e. which is in Ready State) sends its CID number as the response (on the CMD line).
    trace_LOG(trace_DEBUG, "-D- Cmd2()\n\r");
    error = Cmd2(pSd, sdCid);
    if (error) {

        trace_LOG(trace_ERROR, "-E- Error during initialization (%d)\n\r", error);
        return error;
    }

    // Thereafter, the host issues CMD3 (SEND_RELATIVE_ADDR) asks the
    // card to publish a new relative card address (RCA), which is shorter than CID and which is used to
    // address the card in the future data transfer mode. Once the RCA is received the card state changes to
    // the Stand-by State. At this point, if the host wants to assign another RCA number, it can ask the card to
    // publish a new number by sending another CMD3 command to the card. The last published RCA is the
    // actual RCA number of the card.
    trace_LOG(trace_DEBUG, "-D- Cmd3()\n\r");
    error = Cmd3(pSd);
    if (error) {

        trace_LOG(trace_ERROR, "-E- Error during initialization (%d)\n\r", error);
        return error;
    }

    // The host issues SEND_CSD (CMD9) to obtain the Card Specific Data (CSD register),
    // e.g. block length, card storage capacity, etc...
    trace_LOG(trace_DEBUG, "-D- Cmd9()\n\r");
    error = Cmd9(pSd);
    if (error) {

        trace_LOG(trace_ERROR, "-E- Error during initialization (%d)\n\r", error);
        return error;
    }

    // At this stage the Initialization and identification process is achieved
    // The SD card is supposed to be in Stand-by State
    trace_LOG(trace_DEBUG, "-D- Cmd13()\n\r");
    do {
        error = Cmd13(pSd, &status);
        if (error) {

            trace_LOG(trace_ERROR, "-E- Error during initialization (%d)\n\r", error);
            return error;
        }
    }
    while ((status & STATUS_READY_FOR_DATA) == 0);

    // If the 4 bit bus transfer is supported switch to this mode
    // Select the current SD, goto transfer state
    trace_LOG(trace_DEBUG, "-D- Cmd7()\n\r");
    error = Cmd7(pSd, pSd->cardAddress);
    if (error) {

        trace_LOG(trace_ERROR, "-E- Error during initialization (%d)\n\r", error);
        return error;
    }

    if (pSd->cardType != CARD_MMC) {
        // Switch to 4 bits bus width (All SD Card shall support 1-bit, 4 bitswidth
        trace_LOG(trace_DEBUG, "-D- Acmd6()\n\r");
        error = Acmd6(pSd, 4);
        if (error) {

            trace_LOG(trace_ERROR, "-E- Error during initialization (%d)\n\r", error);
            return error;
        }
    }
    else {
        MCI_SetBusWidth(pMci, MCI_SDCBUS_1BIT);
    }

    // In the case of a Standard Capacity SD Memory Card, this command sets the
    // block length (in bytes) for all following block commands (read, write, lock).
    // Default block length is fixed to 512 Bytes.
    // Set length is valid for memory access commands only if partial block read
    // operation are allowed in CSD.
    // In the case of a High Capacity SD Memory Card, block length set by CMD16
    // command does not affect the memory read and write commands. Always 512
    // Bytes fixed block length is used. This command is effective for LOCK_UNLOCK command.
    // In both cases, if block length is set larger than 512Bytes, the card sets the
    // BLOCK_LEN_ERROR bit.
    if (pSd->cardType == CARD_SD) {
        trace_LOG(trace_DEBUG, "-D- Cmd16()\n\r");
        error = Cmd16(pSd, SD_BLOCK_SIZE);
        if (error) {

            trace_LOG(trace_ERROR, "-E- Error during initialization (%d)\n\r", error);
            return error;
        }
    }
    if(pSd->cardType != CARD_SDHC) {
        pSd->totalSize = SD_CSD_TOTAL_SIZE(pSd);
        pSd->blockNr = SD_CSD_BLOCKNR(pSd);
    }
    else {
        pSd->totalSize = SD_CSD_TOTAL_SIZE_HC(pSd);
        pSd->blockNr = SD_CSD_BLOCKNR_HC(pSd);
    }
    // trace_LOG(trace_DEBUG, "TAAC %x, NSAC %x\n\r",SD_CSD_TAAC(pSd),SD_CSD_NSAC(pSd));
    if (pSd->cardType == UNKNOWN_CARD) {

        return SD_ERROR_NOT_INITIALIZED;
    }
    else {
        // trace_LOG(trace_DEBUG, "-D- Card Type: %x\n\r", pSd->cardType);
        return 0;
    }
}

//------------------------------------------------------------------------------
/// Stop the SDcard. This function stops all SD operations.
/// Returns 0 if successful; otherwise returns an SD_ERROR code.
/// \param pSd  Pointer to a SD card driver instance.
/// \param pMci  Pointer to MCI driver already initialized with MCI_Init().
//------------------------------------------------------------------------------
unsigned char SD_Stop(SdCard *pSd, Mci *pMci)
{
    unsigned char error;
    MciCmd *pCommand = &(pSd->command);

    SANITY_CHECK(pSd);
    SANITY_CHECK(pMci);

    if(pCommand->conTrans == MCI_CONTINUE_TRANSFER)
    {
        trace_LOG(trace_DEBUG, "-D- SD_StopTransmission()\n\r");

        error = Cmd12(pSd);
        if(error) {
            return error;
        }
    }

    MCI_Close(pMci);

    return 0;
}
